/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::bessel::i0f::i0f_small;
use crate::common::f_fmla;
use crate::exponents::core_expf;
use crate::logs::fast_logf;
use crate::polyeval::{f_estrin_polyeval7, f_estrin_polyeval8};

/// Modified Bessel of the second kind order 0
///
/// Max ULP 0.5
///
/// This method have exactly one exception which is not correctly rounded with FMA.
pub fn f_k0f(x: f32) -> f32 {
    if x < 0. {
        return f32::NAN;
    }

    if (x.to_bits() & 0x0007_ffff) == 0 {
        if x == 0. {
            return f32::INFINITY;
        }
        if x.is_infinite() {
            return if x.is_sign_positive() { 0. } else { f32::NAN };
        }
        if x.is_nan() {
            return x + x;
        }
    }

    let xb = x.to_bits();

    if xb >= 0x42cbc4fbu32 {
        // 101.88473
        return 0.;
    }

    if xb <= 0x3f800000u32 {
        // 1.0
        return k0f_small(x);
    }

    k0f_asympt(x)
}

/**
K0(x) + log(x) * I0(x) = P(x^2)
hence
K0(x) = P(x^2) - log(x)*I0(x)

Polynomial generated by Wolfram Mathematica:
```text
<<FunctionApproximations`
ClearAll["Global`*"]
f[x_]:=BesselK[0,x]+Log[x]BesselI[0,x]
g[z_]:=f[Sqrt[z]]
{err, approx}=MiniMaxApproximation[g[z],{z,{0.000000001,1},6,0},WorkingPrecision->60]
poly=Numerator[approx][[1]];
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
```
**/
#[inline]
fn k0f_small(x: f32) -> f32 {
    let v_log = fast_logf(x);
    let i0 = i0f_small(x);

    let dx = x as f64;

    let p = f_estrin_polyeval7(
        dx * dx,
        f64::from_bits(0x3fbdadb014541ece),
        f64::from_bits(0x3fd1dadb01453e9c),
        f64::from_bits(0x3f99dadb01491ac7),
        f64::from_bits(0x3f4bb90e82a4f609),
        f64::from_bits(0x3eef4749ebd25b10),
        f64::from_bits(0x3e85d5b5668593af),
        f64::from_bits(0x3e15233b0788618b),
    );
    let c = f_fmla(-i0, v_log, p);
    c as f32
}

/**
Generated in Wolfram

Computes sqrt(x)*exp(x)*K0(x)=Pn(1/x)/Qm(1/x)
hence
K0(x) = Pn(1/x)/Qm(1/x) / (sqrt(x) * exp(x))

```text
<<FunctionApproximations`
ClearAll["Global`*"]
f[x_]:=Sqrt[x] Exp[x] BesselK[0,x]
g[z_]:=f[1/z]
{err, approx}=MiniMaxApproximation[g[z],{z,{0.0000000000001,1},7,7},WorkingPrecision->60]
poly=Numerator[approx][[1]];
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
poly=Denominator[approx][[1]];
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
```
**/
#[inline]
fn k0f_asympt(x: f32) -> f32 {
    let dx = x as f64;
    let recip = 1. / dx;
    let e = core_expf(x);
    let r_sqrt = dx.sqrt();

    let p_num = f_estrin_polyeval8(
        recip,
        f64::from_bits(0x3ff40d931ff62701),
        f64::from_bits(0x402d8410a62d9c17),
        f64::from_bits(0x404e9f1804dd7e54),
        f64::from_bits(0x405c076822dcd255),
        f64::from_bits(0x4057379c6932949f),
        f64::from_bits(0x403ffd64a0bd54b7),
        f64::from_bits(0x400cc53ed733fd97),
        f64::from_bits(0x3faf8cc8756944eb),
    );
    let p_den = f_estrin_polyeval8(
        recip,
        f64::from_bits(0x3ff0000000000000),
        f64::from_bits(0x4027ccde1d27ffc9),
        f64::from_bits(0x40492418136fb90f),
        f64::from_bits(0x4057be8a00983906),
        f64::from_bits(0x4054cc77d2379b76),
        f64::from_bits(0x403fd218713ec08d),
        f64::from_bits(0x4011c77649d3f65f),
        f64::from_bits(0x3fc2080a59e87324),
    );
    let v = p_num / p_den;
    let pp = v / (e * r_sqrt);
    pp as f32
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_k0f() {
        assert_eq!(f_k0f(2.034804e-5), 10.918458);
        assert_eq!(f_k0f(0.010260499), 4.695535);
        assert_eq!(f_k0f(0.3260499), 1.2965646);
        assert_eq!(f_k0f(0.72341), 0.636511734);
        assert_eq!(f_k0f(0.), f32::INFINITY);
        assert_eq!(f_k0f(-0.), f32::INFINITY);
        assert!(f_k0f(-0.5).is_nan());
        assert!(f_k0f(f32::NEG_INFINITY).is_nan());
        assert_eq!(f_k0f(f32::INFINITY), 0.);
    }
}
